<?php

namespace App\Console\Commands;

use App\Events\FoodStatusChanged;
use App\Models\Kot;
use Illuminate\Console\Command;

class TestFoodStatusNotification extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'test:food-status-notification 
                           {kotId : The ID of the KOT to test with}
                           {status=food_ready : The status to change to (in_kitchen, food_ready, served)}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Test the food status notification system by triggering a status change event';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $kotId = $this->argument('kotId');
        $newStatus = $this->argument('status');

        // Validate status
        $validStatuses = ['in_kitchen', 'food_ready', 'served'];
        if (!in_array($newStatus, $validStatuses)) {
            $this->error("Invalid status. Valid statuses are: " . implode(', ', $validStatuses));
            return 1;
        }

        // Find the KOT
        $kot = Kot::with(['order.waiter', 'order.table', 'items.menuItem'])->find($kotId);
        if (!$kot) {
            $this->error("KOT with ID {$kotId} not found.");
            return 1;
        }

        // Store the previous status
        $previousStatus = $kot->status;

        $this->info("Testing food status notification...");
        $this->line("KOT ID: {$kot->id}");
        $this->line("Order Number: {$kot->order->order_number}");
        $this->line("Table: {$kot->order->table?->table_code ?? 'Takeaway'}");
        $this->line("Previous Status: {$previousStatus}");
        $this->line("New Status: {$newStatus}");

        // Check if status is actually changing
        if ($previousStatus === $newStatus) {
            $this->warn("Status is not changing (already {$newStatus}). Continuing with test anyway...");
        }

        // Update the KOT status
        $kot->update(['status' => $newStatus]);
        $this->info("✓ KOT status updated");

        // Dispatch the event - DISABLED: Using simple session-based notifications instead
        try {
            // event(new FoodStatusChanged($kot->fresh(), $previousStatus, $newStatus));
            $this->info("✓ Event dispatch skipped (using session-based notifications)");
        } catch (\Exception $e) {
            $this->error("✗ Failed to dispatch event: " . $e->getMessage());
            return 1;
        }

        $this->newLine();
        $this->info("Test completed successfully!");
        $this->line("Check the following:");
        $this->line("1. Pusher push notifications should be sent to relevant users");
        $this->line("2. Real-time notifications should appear in the dashboard");
        $this->line("3. If status is 'food_ready', sound should play and popup should appear");
        $this->line("4. Check the logs for any errors: tail -f storage/logs/laravel.log");

        return 0;
    }
}