<?php

namespace App\Listeners;

use App\Events\FoodStatusChanged;
use App\Http\Controllers\DashboardController;
use App\Models\User;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class FoodStatusChangedListener implements ShouldQueue
{
    use InteractsWithQueue;

    /**
     * Create the event listener.
     */
    public function __construct()
    {
        //
    }

    /**
     * Handle the event.
     */
    public function handle(FoodStatusChanged $event): void
    {
        try {
            // Only send push notifications for important status changes
            if (!in_array($event->newStatus, ['food_ready', 'served'])) {
                return;
            }

            // Get restaurant users who should receive notifications
            $notificationUsers = $this->getNotificationUsers($event);

            if ($notificationUsers->isEmpty()) {
                return;
            }

            // Prepare notification data
            $title = $this->getNotificationTitle($event);
            $body = $event->orderData['message'] ?? $this->getStatusMessage($event);
            $link = $this->getNotificationLink($event);

            // Send push notifications
            $this->sendPushNotifications($notificationUsers, $title, $body, $link);

            // Log successful notification
            Log::info('Food status notification sent', [
                'kot_id' => $event->kot->id,
                'status' => $event->newStatus,
                'users_count' => $notificationUsers->count(),
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to send food status notification', [
                'error' => $e->getMessage(),
                'kot_id' => $event->kot->id,
                'status' => $event->newStatus,
            ]);
        }
    }

    /**
     * Get users who should receive notifications
     */
    private function getNotificationUsers(FoodStatusChanged $event)
    {
        $restaurantId = $event->orderData['restaurant_id'];
        $waiterId = $event->orderData['waiter_id'];

        // Get users who should be notified based on role and assignment
        $query = User::where('restaurant_id', $restaurantId)
            ->where('active', 1);

        // For food_ready status, prioritize the assigned waiter
        if ($event->newStatus === 'food_ready' && $waiterId) {
            // First try to notify the assigned waiter
            $assignedWaiter = User::where('id', $waiterId)
                ->where('restaurant_id', $restaurantId)
                ->where('active', 1)
                ->first();

            if ($assignedWaiter) {
                return collect([$assignedWaiter]);
            }
        }

        // If no specific waiter or waiter not available, notify managers and waiters
        return $query->whereHas('roles', function ($roleQuery) {
            $roleQuery->whereIn('name', ['Restaurant Admin', 'Waiter', 'Manager']);
        })->get();
    }

    /**
     * Get notification title based on status
     */
    private function getNotificationTitle(FoodStatusChanged $event): string
    {
        switch ($event->newStatus) {
            case 'food_ready':
                return '🍽️ Food Ready for Serving!';
            case 'served':
                return '✅ Order Served';
            default:
                return '📋 Kitchen Update';
        }
    }

    /**
     * Get deep link for notification
     */
    private function getNotificationLink(FoodStatusChanged $event): string
    {
        // Link to the KOT page or orders page
        return route('kots.index') . '?highlight=' . $event->kot->id;
    }

    /**
     * Get user-friendly status message
     */
    private function getStatusMessage(FoodStatusChanged $event): string
    {
        switch ($event->newStatus) {
            case 'food_ready':
                return "Table {$event->orderData['table_code']} - Order #{$event->orderData['order_number']} is ready for serving!";
            case 'served':
                return "Table {$event->orderData['table_code']} - Order #{$event->orderData['order_number']} has been served.";
            case 'in_kitchen':
                return "Table {$event->orderData['table_code']} - Order #{$event->orderData['order_number']} is being prepared.";
            default:
                return "Status updated for Table {$event->orderData['table_code']} - Order #{$event->orderData['order_number']}.";
        }
    }

    /**
     * Send push notifications using Pusher Beams
     */
    private function sendPushNotifications($users, string $title, string $body, string $link): void
    {
        if (!pusherSettings()->beamer_status || $users->isEmpty()) {
            return;
        }

        try {
            $beamsClient = new \Pusher\PushNotifications\PushNotifications([
                'instanceId' => pusherSettings()->instance_id,
                'secretKey' => pusherSettings()->beam_secret,
            ]);

            // Prepare user IDs for push notifications
            $pushUserIds = $users->map(function ($user) {
                return Str::slug(global_setting()->name) . '-' . $user->id;
            })->toArray();

            // Send the notification
            $publishResponse = $beamsClient->publishToUsers(
                $pushUserIds,
                [
                    'web' => [
                        'notification' => [
                            'title' => $title,
                            'body' => $body,
                            'deep_link' => $link,
                            'icon' => global_setting()->logo_url,
                            'badge' => global_setting()->logo_url,
                            'requireInteraction' => true, // Keep notification visible until user interacts
                            'tag' => 'food-status-' . time(), // Prevent duplicate notifications
                            'data' => [
                                'url' => $link,
                                'priority' => 'high',
                            ],
                        ],
                    ],
                ]
            );

            Log::info('Push notification sent successfully', [
                'users_count' => count($pushUserIds),
                'response' => $publishResponse,
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to send push notification', [
                'error' => $e->getMessage(),
                'users_count' => $users->count(),
            ]);
        }
    }
}