<?php

namespace App\Console\Commands;

use App\Models\Kot;
use Illuminate\Console\Command;

class TestFoodReadyNotification extends Command
{
    protected $signature = 'test:food-ready {kotId? : Optional KOT ID to test with}';
    protected $description = 'Test food ready notification system by resetting session count';

    public function handle()
    {
        $kotId = $this->argument('kotId');
        
        if ($kotId) {
            // Test with specific KOT
            $kot = Kot::find($kotId);
            if (!$kot) {
                $this->error("KOT with ID {$kotId} not found.");
                return 1;
            }
            
            $this->info("Testing with KOT ID: {$kotId}");
            $this->line("Current status: {$kot->status}");
            
            if ($kot->status !== 'food_ready') {
                $this->warn("KOT is not in 'food_ready' status. Changing it now...");
                $kot->update(['status' => 'food_ready']);
                $this->info("Status updated to 'food_ready'");
            }
        }
        
        // Get current count of food ready orders
        $currentCount = Kot::where('status', 'food_ready')->count();
        $this->info("Current food_ready KOTs: {$currentCount}");
        
        // Check current session
        $sessionCount = session('food_ready_count', 'not_set');
        $this->info("Current session count: {$sessionCount}");
        
        // Reset session to trigger notification
        session(['food_ready_count' => $currentCount - 1]);
        $this->info("Reset session count to: " . ($currentCount - 1));
        
        $this->newLine();
        $this->info("✅ Test setup complete!");
        $this->line("Now watch the waiter/admin interface for:");
        $this->line("1. Popup notification within 15 seconds");
        $this->line("2. Sound alert playing");
        $this->line("3. Check browser console for debug messages");
        $this->line("4. Check logs: tail -f storage/logs/laravel.log");
        
        return 0;
    }
}