<?php

namespace App\Events;

use App\Models\Kot;
use Illuminate\Broadcasting\Channel;
use Illuminate\Broadcasting\InteractsWithSockets;
use Illuminate\Broadcasting\PrivateChannel;
use Illuminate\Contracts\Broadcasting\ShouldBroadcast;
use Illuminate\Foundation\Events\Dispatchable;
use Illuminate\Queue\SerializesModels;

class FoodStatusChanged implements ShouldBroadcast
{
    use Dispatchable, InteractsWithSockets, SerializesModels;

    public $kot;
    public $previousStatus;
    public $newStatus;
    public $orderData;

    /**
     * Create a new event instance.
     */
    public function __construct(Kot $kot, string $previousStatus, string $newStatus)
    {
        $this->kot = $kot;
        $this->previousStatus = $previousStatus;
        $this->newStatus = $newStatus;
        
        // Load relationships for broadcasting
        $this->kot->load(['order.waiter', 'order.table', 'items.menuItem']);
        
        // Prepare data for broadcasting
        $this->orderData = [
            'kot_id' => $this->kot->id,
            'kot_number' => $this->kot->kot_number,
            'order_id' => $this->kot->order_id,
            'order_number' => $this->kot->order->order_number,
            'table_code' => $this->kot->order->table?->table_code ?? 'Takeaway',
            'table_id' => $this->kot->order->table_id,
            'waiter_id' => $this->kot->order->waiter_id,
            'waiter_name' => $this->kot->order->waiter?->name ?? 'Unassigned',
            'branch_id' => $this->kot->branch_id,
            'restaurant_id' => $this->kot->order->branch?->restaurant_id,
            'previous_status' => $this->previousStatus,
            'new_status' => $this->newStatus,
            'items_count' => $this->kot->items->count(),
            'created_at' => $this->kot->created_at->toISOString(),
            'updated_at' => now()->toISOString(),
        ];
    }

    /**
     * Get the channels the event should broadcast on.
     *
     * @return array<int, \Illuminate\Broadcasting\Channel>
     */
    public function broadcastOn(): array
    {
        return [
            new PrivateChannel('restaurant.' . $this->orderData['restaurant_id']),
            new PrivateChannel('branch.' . $this->orderData['branch_id']),
            new PrivateChannel('waiter.' . $this->orderData['waiter_id']),
        ];
    }

    /**
     * Get the data to broadcast.
     *
     * @return array
     */
    public function broadcastWith(): array
    {
        return [
            'type' => 'food_status_changed',
            'data' => $this->orderData,
            'message' => $this->getStatusMessage(),
            'priority' => $this->getNotificationPriority(),
            'sound_enabled' => $this->shouldPlaySound(),
            'timestamp' => now()->timestamp,
        ];
    }

    /**
     * Get the broadcast event name.
     */
    public function broadcastAs(): string
    {
        return 'food.status.changed';
    }

    /**
     * Get user-friendly status message
     */
    private function getStatusMessage(): string
    {
        switch ($this->newStatus) {
            case 'food_ready':
                return "Table {$this->orderData['table_code']} - Order #{$this->orderData['order_number']} is ready for serving!";
            case 'served':
                return "Table {$this->orderData['table_code']} - Order #{$this->orderData['order_number']} has been served.";
            case 'in_kitchen':
                return "Table {$this->orderData['table_code']} - Order #{$this->orderData['order_number']} is being prepared.";
            default:
                return "Status updated for Table {$this->orderData['table_code']} - Order #{$this->orderData['order_number']}.";
        }
    }

    /**
     * Get notification priority level
     */
    private function getNotificationPriority(): string
    {
        switch ($this->newStatus) {
            case 'food_ready':
                return 'high'; // This needs immediate waiter attention
            case 'served':
                return 'medium'; // Good to know but less urgent
            case 'in_kitchen':
                return 'low'; // Information only
            default:
                return 'medium';
        }
    }

    /**
     * Determine if sound should be played
     */
    private function shouldPlaySound(): bool
    {
        // Play sound for high priority notifications (food ready)
        return $this->newStatus === 'food_ready';
    }
}